C      *****************************************************************
C      * Subroutine FluxErr                                            *
C      * Checks                                                        *
C      * 1. that the water leaving each layer is not greater than the  *
C      *    amount of water in that layer, we don't allow water        *
C      *    entering the layer to move through the layer in one time   *
C      *    step,                                                      *
C      * 2. that the flux for the bottom layer is not moving up.  The  *
C      *    model assumes that no nutrients are entering the bottom    *
C      *    layer from below because we have no way of knowing the     *
C      *    concentrations of nutrients that would be moving up with   *
C      *    the water.                                                 *
C      * If an error is found a descriptive error message is output and*
C      * a flag is returned to the calling routine to indicate that an *
C      * error was found.                                              *
C      *                                                               *
C      * Variables:                                                    *
C      * CatchArea -Input - A vector, of size NLayers, of the catchment*
C      *                    area for each layer.  Input from TREGRO, or*
C      *                    read into the model by the routine InParam.*
C      *                    Units are m**2.                            *
C      *                    (Common block VModel, file VModel.f)       *
C      * ElapT     -Input - The elapsed time since the last call, i.e.,*
C      *                    the model timestep.  (This is model time,  *
C      *                    not real time.)  Input from TREGRO, or read*
C      *                    into the model by the routine InParam.     *
C      *                    Units are s.                               *
C      *                    (Common block VModel, file VModel.f)       *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * L         -Local - Layer index.  Unitless.                    *
C      * LPerM3    -Local - Constant for converting m**3 to liters.    *
C      *                    Units are L/m**3.                          *
C      * NLayers   -Input - The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * OldTheta  -Input - A vector, of size NLayers, of volumetric   *
C      *                    soil moisture for each layer, from the     *
C      *                    previous time step. Units are              *
C      *                    cm**3/cm**3.                               *
C      *                    (Common block VModel, file VModel.f)       *
C      * SoilWat   -Local - The amount of soil water.  Units are L.    *
C      * Thickness -Input - A vector, of size NLayers, of thicknesses  *
C      *                    of each soil layer.  Input from TREGRO, or *
C      *                    read into the model by the routine InParam.*
C      *                    Units are m.                               *
C      *                    (Common block VModel, file VModel.f)       *
C      * WFlux     -Input - A vector, of size NLayers, of net water    *
C      *                    fluxes between layers.  Flux(0) is the net *
C      *                    flux between the atmosphere and the top    *
C      *                    layer, Flux(1) is the net flux between     *
C      *                    layers number 1 and 2, Flux(3) is the net  *
C      *                    flux between layers number 3 and 4.  If    *
C      *                    the flux is positive, the water is moving  *
C      *                    down; if the flux is negative, the water is*
C      *                    moving up.  Input from TREGRO, or read into*
C      *                    the model by the routine InParam.  Units   *
C      *                    are L/(m**2 sec).                          *                     
C      *                    (Common block VModel, file VModel.f)       *
C      * YASOut    -Input - The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE FLUXERR(GOAHEAD)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							REAL*8 NETFLUX, SOILWAT
							INTEGER GOAHEAD, L, LPERM3/1000/
							
C      *-----------------------------------------*
C      * Indicate that no errors have been found *
C      *-----------------------------------------*
       GOAHEAD = 1

C      *----------------------*
C      * Loop over all layers *
C      *----------------------*
       DO 100 L=1,NLAYERS

C         *------------------------------------------------*
C         * Calculate the soil water for the current layer *
C         *------------------------------------------------*
          SOILWAT = CATCHAREA(L) * THICKNESS(L) * OLDTHETA(L) * LPERM3

C         *------------------------------------------------------*
C         * The amount of water leaving the layer is the amount  *
C         * going out of the bottom and/or the amount going out  *
C         * of the top, in liters.                               *
C         *------------------------------------------------------*
          NETFLUX = 0.0
          IF (WFLUX(L).GT.0) THEN
	            NETFLUX = NETFLUX + WFLUX(L) * CATCHAREA(L) * ELAPT
	         ENDIF
	         IF (WFLUX(L-1).LT.0) THEN
	            NETFLUX = NETFLUX + ABS(WFLUX(L-1)) * CATCHAREA(L) * 
     >                 ELAPT
	         ENDIF

C         *------------------------------------------------------*
C         * If the amount leaving the layer is more than there   *
C         * is in the layer, output an error message and set the *
C         * error indicator flag.  We do not allow the water     *
C         * entering to pass right through in one time step.     *
C         *------------------------------------------------------*
          IF (NETFLUX.GT.SOILWAT) THEN
	            WRITE(6,1101) L, NETFLUX, SOILWAT
 	           WRITE(YASOUT,1101) L, NETFLUX, SOILWAT
1101         FORMAT(' ', 'More water is leaving layer ',I2,
     >       ' than is there',/,' ','Net flux = ',F10.5,
     >       ' (L)    Soil Water = ',F10.5, ' (L)', //, ' ', 
     >       'Consider using a shorter timestep or a thicker ',
     >       'soil layer ',
     >       'Routine: FluxErr')
	            GOAHEAD = 0
	         ENDIF
	   
  100  CONTINUE
  
C      *-------------------------------------------------------------------*
C      * Check to see if the last layer has water moving up into it.       *
C      * If it does, output an error message and set error indicator flag  *
C      *-------------------------------------------------------------------*
       IF(WFLUX(NLAYERS).LT.0) THEN
	         WRITE(6,1111) NLAYERS, WFLUX(NLAYERS)
	         WRITE(YASOUT,1111) NLAYERS, WFLUX(NLAYERS)
 1111     FORMAT(' ','Layer ',I2,' has water moving up into it',/,' ',
     >    'The amount of water is: ',F10.5,' L/(m**2-sec)',//,' ',
     >    'Routine: FluxErr')
		        GOAHEAD = 0
	      ENDIF

       RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
